@extends('layouts.app')

@section('title', 'Recargar Saldo')

@section('content')
<div class="min-h-[calc(100vh-140px)] flex items-center justify-center p-4" x-data="recargaPage()">
    <div class="max-w-2xl w-full">
        <!-- Balance actual -->
        <div class="glass-effect rounded-2xl p-4 sm:p-6 mb-6 border border-cyan-500/30 bg-gradient-to-r from-cyan-500/10 to-blue-500/10">
            <div class="flex flex-col sm:flex-row items-start sm:items-center justify-between gap-4">
                <div class="flex items-center gap-3 sm:gap-4">
                    <div class="w-12 h-12 sm:w-16 sm:h-16 rounded-full bg-gradient-to-br from-cyan-400 to-blue-500 flex items-center justify-center shadow-lg shadow-cyan-500/30">
                        <span class="text-2xl sm:text-3xl">💰</span>
                    </div>
                    <div>
                        <p class="text-xs sm:text-sm text-cyan-400/80 uppercase tracking-wider font-semibold">Tu Saldo Actual</p>
                        <p class="text-3xl sm:text-4xl font-black text-cyan-400">${{ number_format($balance, 0) }}</p>
                    </div>
                </div>
                <a href="{{ route('bingo.index') }}" 
                   title="Volver a la pantalla principal del bingo"
                   aria-label="Volver al inicio"
                   class="w-full sm:w-auto text-center px-4 py-2 rounded-lg bg-gray-700 text-gray-300 hover:bg-gray-600 transition-colors text-sm font-semibold">
                    ← Volver al Inicio
                </a>
            </div>
        </div>

        <!-- Título -->
        <div class="text-center mb-6 sm:mb-8">
            <h1 class="text-2xl sm:text-3xl font-black text-white mb-2">💳 Recargar Saldo</h1>
            <p class="text-sm sm:text-base text-gray-400">Elige un monto y método de pago</p>
        </div>

        <!-- Montos predefinidos -->
        <div class="glass-effect rounded-2xl p-4 sm:p-6 mb-6">
            <p class="text-xs sm:text-sm font-semibold text-yellow-400 mb-3 sm:mb-4 uppercase tracking-wider">Selecciona un monto</p>
            <div class="grid grid-cols-2 md:grid-cols-4 gap-2 sm:gap-3">
                <template x-for="monto in montosPredefinidos" :key="monto">
                    <button @click="montoSeleccionado = monto; montoPersonalizado = ''"
                            class="p-3 sm:p-4 rounded-xl border-2 transition-all duration-200 transform hover:scale-105"
                            :class="montoSeleccionado === monto 
                                ? 'border-yellow-500 bg-yellow-500/20 shadow-lg shadow-yellow-500/30' 
                                : 'border-gray-600 bg-gray-800/50 hover:border-gray-500'">
                        <p class="text-xl sm:text-2xl font-black" :class="montoSeleccionado === monto ? 'text-yellow-400' : 'text-white'">
                            $<span x-text="(monto / 1000) + 'K'"></span>
                        </p>
                        <p class="text-[10px] sm:text-xs text-gray-400 mt-1">
                            <span x-text="Math.floor(monto / 1000)"></span> cartones
                        </p>
                    </button>
                </template>
            </div>

            <!-- Monto personalizado -->
            <div class="mt-4">
                <label class="block text-sm text-gray-400 mb-2">O ingresa un monto personalizado:</label>
                <div class="relative">
                    <span class="absolute left-4 top-1/2 -translate-y-1/2 text-gray-400 font-bold">$</span>
                    <input type="number" 
                           x-model="montoPersonalizado" 
                           @input="montoSeleccionado = null"
                           placeholder="50,000"
                           min="1000"
                           step="1000"
                           class="w-full pl-8 pr-4 py-3 rounded-xl bg-gray-800 border border-gray-600 text-white font-bold text-lg focus:border-yellow-500 focus:outline-none focus:ring-2 focus:ring-yellow-500/20">
                </div>
            </div>
        </div>

        <!-- Métodos de pago -->
        <div class="glass-effect rounded-2xl p-4 sm:p-6 mb-6">
            <p class="text-xs sm:text-sm font-semibold text-yellow-400 mb-3 sm:mb-4 uppercase tracking-wider">Método de Pago</p>
            <div class="grid grid-cols-1 sm:grid-cols-3 gap-3 sm:gap-4">
                <!-- Stripe (Tarjeta) -->
                <button @click="metodoPago = 'stripe'"
                        title="Pagar con tarjeta de crédito o débito (Visa, Mastercard)"
                        aria-label="Seleccionar pago con tarjeta"
                        class="p-4 rounded-xl border-2 transition-all duration-200 text-left"
                        :class="metodoPago === 'stripe' 
                            ? 'border-blue-500 bg-blue-500/20' 
                            : 'border-gray-600 bg-gray-800/50 hover:border-gray-500'">
                    <div class="flex items-center gap-3 mb-2">
                        <div class="w-10 h-10 rounded-lg bg-gradient-to-br from-blue-500 to-indigo-600 flex items-center justify-center">
                            <svg class="w-6 h-6 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 10h18M7 15h1m4 0h1m-7 4h12a3 3 0 003-3V8a3 3 0 00-3-3H6a3 3 0 00-3 3v8a3 3 0 003 3z"/>
                            </svg>
                        </div>
                        <div>
                            <p class="font-bold text-white">Tarjeta</p>
                            <p class="text-xs text-gray-400">Visa, Mastercard</p>
                        </div>
                    </div>
                    <div class="flex gap-1">
                        <div class="w-8 h-5 bg-blue-600 rounded flex items-center justify-center text-[8px] font-bold text-white">VISA</div>
                        <div class="w-8 h-5 bg-red-500 rounded flex items-center justify-center">
                            <div class="w-2 h-2 bg-yellow-400 rounded-full"></div>
                            <div class="w-2 h-2 bg-red-600 rounded-full -ml-1"></div>
                        </div>
                    </div>
                </button>

                <!-- PayPal -->
                <button @click="metodoPago = 'paypal'"
                        title="Pagar con tu cuenta de PayPal de forma segura"
                        aria-label="Seleccionar pago con PayPal"
                        class="p-4 rounded-xl border-2 transition-all duration-200 text-left"
                        :class="metodoPago === 'paypal' 
                            ? 'border-blue-400 bg-blue-400/20' 
                            : 'border-gray-600 bg-gray-800/50 hover:border-gray-500'">
                    <div class="flex items-center gap-3 mb-2">
                        <div class="w-10 h-10 rounded-lg bg-gradient-to-br from-blue-400 to-blue-600 flex items-center justify-center">
                            <span class="text-white font-black text-sm">PP</span>
                        </div>
                        <div>
                            <p class="font-bold text-white">PayPal</p>
                            <p class="text-xs text-gray-400">Pago seguro</p>
                        </div>
                    </div>
                    <div class="w-16 h-5 bg-[#003087] rounded flex items-center justify-center">
                        <span class="text-[8px] font-bold text-white">PayPal</span>
                    </div>
                </button>

                <!-- Bitcoin -->
                <button @click="metodoPago = 'bitcoin'"
                        title="Pagar con criptomonedas (Bitcoin, Ethereum, USDT)"
                        aria-label="Seleccionar pago con Bitcoin"
                        class="p-4 rounded-xl border-2 transition-all duration-200 text-left"
                        :class="metodoPago === 'bitcoin' 
                            ? 'border-orange-500 bg-orange-500/20' 
                            : 'border-gray-600 bg-gray-800/50 hover:border-gray-500'">
                    <div class="flex items-center gap-3 mb-2">
                        <div class="w-10 h-10 rounded-lg bg-gradient-to-br from-orange-400 to-orange-600 flex items-center justify-center">
                            <span class="text-white font-black text-lg">₿</span>
                        </div>
                        <div>
                            <p class="font-bold text-white">Bitcoin</p>
                            <p class="text-xs text-gray-400">Crypto</p>
                        </div>
                    </div>
                    <div class="flex gap-1">
                        <div class="w-6 h-5 bg-orange-500 rounded flex items-center justify-center text-[10px] font-bold text-white">BTC</div>
                        <div class="w-6 h-5 bg-blue-500 rounded flex items-center justify-center text-[10px] font-bold text-white">ETH</div>
                        <div class="w-6 h-5 bg-green-500 rounded flex items-center justify-center text-[10px] font-bold text-white">USDT</div>
                    </div>
                </button>
            </div>
        </div>

        <!-- Resumen y botón de pago -->
        <div class="glass-effect rounded-2xl p-6">
            <div class="flex items-center justify-between mb-4">
                <div>
                    <p class="text-sm text-gray-400">Total a pagar</p>
                    <p class="text-3xl font-black text-white">$<span x-text="getMontoFinal().toLocaleString()"></span></p>
                </div>
                <div class="text-right">
                    <p class="text-sm text-gray-400">Recibirás</p>
                    <p class="text-2xl font-bold text-green-400">$<span x-text="getMontoFinal().toLocaleString()"></span></p>
                    <p class="text-xs text-gray-500">+ <span x-text="getBonus()"></span>% bonus</p>
                </div>
            </div>

            <button @click="procesarPago()"
                    :disabled="!canProceed() || procesando"
                    :title="canProceed() ? 'Proceder al pago con ' + getMetodoNombre() : 'Selecciona monto y método de pago'"
                    aria-label="Procesar pago"
                    class="w-full py-4 rounded-xl font-bold text-lg uppercase tracking-wider transition-all transform hover:scale-[1.02] active:scale-[0.98] disabled:opacity-50 disabled:cursor-not-allowed disabled:hover:scale-100"
                    :class="metodoPago === 'stripe' ? 'bg-gradient-to-r from-blue-500 to-indigo-600 text-white' :
                            metodoPago === 'paypal' ? 'bg-gradient-to-r from-blue-400 to-blue-600 text-white' :
                            metodoPago === 'bitcoin' ? 'bg-gradient-to-r from-orange-500 to-orange-600 text-white' :
                            'bg-gray-600 text-gray-400'">
                <span x-show="!procesando" class="flex items-center justify-center gap-2">
                    <template x-if="metodoPago === 'stripe'">
                        <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 10h18M7 15h1m4 0h1m-7 4h12a3 3 0 003-3V8a3 3 0 00-3-3H6a3 3 0 00-3 3v8a3 3 0 003 3z"/>
                        </svg>
                    </template>
                    <template x-if="metodoPago === 'paypal'">
                        <span class="font-black">PP</span>
                    </template>
                    <template x-if="metodoPago === 'bitcoin'">
                        <span class="font-black">₿</span>
                    </template>
                    <span x-text="metodoPago ? 'Pagar con ' + getMetodoNombre() : 'Selecciona un método'"></span>
                </span>
                <span x-show="procesando" class="flex items-center justify-center gap-2">
                    <svg class="w-5 h-5 animate-spin" fill="none" viewBox="0 0 24 24">
                        <circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4"></circle>
                        <path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                    </svg>
                    Procesando...
                </span>
            </button>

            <p class="text-xs text-gray-500 text-center mt-4">
                🔒 Transacción segura • Los pagos son procesados de forma segura
            </p>
        </div>

        <!-- Historial de transacciones -->
        @if(count($transacciones) > 0)
        <div class="glass-effect rounded-2xl p-6 mt-6">
            <p class="text-sm font-semibold text-gray-400 mb-4 uppercase tracking-wider">Últimas Transacciones</p>
            <div class="space-y-2">
                @foreach(array_slice(array_reverse($transacciones), 0, 5) as $tx)
                <div class="flex items-center justify-between p-3 rounded-lg bg-gray-800/50">
                    <div class="flex items-center gap-3">
                        <div class="w-8 h-8 rounded-full flex items-center justify-center
                            {{ $tx['tipo'] === 'recarga' ? 'bg-green-500/20 text-green-400' : 'bg-red-500/20 text-red-400' }}">
                            {{ $tx['tipo'] === 'recarga' ? '↓' : '↑' }}
                        </div>
                        <div>
                            <p class="text-sm font-semibold text-white">{{ ucfirst($tx['tipo']) }}</p>
                            <p class="text-xs text-gray-400">{{ $tx['metodo'] ?? 'Sistema' }} • {{ $tx['fecha'] }}</p>
                        </div>
                    </div>
                    <p class="font-bold {{ $tx['tipo'] === 'recarga' ? 'text-green-400' : 'text-red-400' }}">
                        {{ $tx['tipo'] === 'recarga' ? '+' : '-' }}${{ number_format($tx['monto'], 0) }}
                    </p>
                </div>
                @endforeach
            </div>
        </div>
        @endif
    </div>

    <!-- Modal de pago con tarjeta (Stripe) -->
    <div x-cloak x-show="showStripeModal" 
         x-transition:enter="transition ease-out duration-300"
         x-transition:enter-start="opacity-0"
         x-transition:enter-end="opacity-100"
         class="fixed inset-0 z-50 flex items-center justify-center p-4 bg-black/80"
         @click.self="showStripeModal = false">
        <div class="glass-effect rounded-2xl p-6 max-w-md w-full">
            <div class="flex items-center justify-between mb-6">
                <h3 class="text-xl font-bold text-white">💳 Pago con Tarjeta</h3>
                <button @click="showStripeModal = false" class="text-gray-400 hover:text-white">✕</button>
            </div>
            
            <form @submit.prevent="confirmarPagoStripe()">
                <div class="space-y-4">
                    <div>
                        <label class="block text-sm text-gray-400 mb-1">Número de tarjeta</label>
                        <input type="text" x-model="tarjeta.numero" placeholder="4242 4242 4242 4242" maxlength="19"
                               @input="formatearTarjeta()"
                               class="w-full px-4 py-3 rounded-lg bg-gray-800 border border-gray-600 text-white focus:border-blue-500 focus:outline-none">
                    </div>
                    <div class="grid grid-cols-2 gap-4">
                        <div>
                            <label class="block text-sm text-gray-400 mb-1">Vencimiento</label>
                            <input type="text" x-model="tarjeta.vencimiento" placeholder="MM/AA" maxlength="5"
                                   class="w-full px-4 py-3 rounded-lg bg-gray-800 border border-gray-600 text-white focus:border-blue-500 focus:outline-none">
                        </div>
                        <div>
                            <label class="block text-sm text-gray-400 mb-1">CVV</label>
                            <input type="text" x-model="tarjeta.cvv" placeholder="123" maxlength="4"
                                   class="w-full px-4 py-3 rounded-lg bg-gray-800 border border-gray-600 text-white focus:border-blue-500 focus:outline-none">
                        </div>
                    </div>
                    <div>
                        <label class="block text-sm text-gray-400 mb-1">Nombre en la tarjeta</label>
                        <input type="text" x-model="tarjeta.nombre" placeholder="JUAN PEREZ"
                               class="w-full px-4 py-3 rounded-lg bg-gray-800 border border-gray-600 text-white focus:border-blue-500 focus:outline-none uppercase">
                    </div>
                </div>
                
                <div class="mt-6 p-4 rounded-lg bg-blue-500/10 border border-blue-500/30">
                    <div class="flex justify-between text-sm">
                        <span class="text-gray-400">Total a pagar:</span>
                        <span class="font-bold text-white">$<span x-text="getMontoFinal().toLocaleString()"></span></span>
                    </div>
                </div>
                
                <button type="submit" :disabled="procesandoPago"
                        class="w-full mt-4 py-3 rounded-xl font-bold bg-gradient-to-r from-blue-500 to-indigo-600 text-white hover:from-blue-400 hover:to-indigo-500 disabled:opacity-50">
                    <span x-show="!procesandoPago">Confirmar Pago</span>
                    <span x-show="procesandoPago">Procesando...</span>
                </button>
            </form>
            
            <p class="text-xs text-gray-500 text-center mt-4">
                🔒 Demo: usa 4242 4242 4242 4242 para simular
            </p>
        </div>
    </div>

    <!-- Modal de PayPal -->
    <div x-cloak x-show="showPaypalModal" 
         x-transition:enter="transition ease-out duration-300"
         x-transition:enter-start="opacity-0"
         x-transition:enter-end="opacity-100"
         class="fixed inset-0 z-50 flex items-center justify-center p-4 bg-black/80"
         @click.self="showPaypalModal = false">
        <div class="glass-effect rounded-2xl p-6 max-w-md w-full text-center">
            <div class="w-20 h-20 mx-auto mb-4 rounded-full bg-gradient-to-br from-blue-400 to-blue-600 flex items-center justify-center">
                <span class="text-white font-black text-2xl">PP</span>
            </div>
            <h3 class="text-xl font-bold text-white mb-2">PayPal</h3>
            <p class="text-gray-400 mb-6">Serás redirigido a PayPal para completar el pago de $<span x-text="getMontoFinal().toLocaleString()"></span></p>
            
            <button @click="confirmarPagoPaypal()" :disabled="procesandoPago"
                    class="w-full py-3 rounded-xl font-bold bg-[#0070ba] text-white hover:bg-[#003087] disabled:opacity-50">
                <span x-show="!procesandoPago">Continuar con PayPal</span>
                <span x-show="procesandoPago">Conectando...</span>
            </button>
            
            <button @click="showPaypalModal = false" class="w-full mt-2 py-2 text-gray-400 hover:text-white">
                Cancelar
            </button>
        </div>
    </div>

    <!-- Modal de Bitcoin -->
    <div x-cloak x-show="showBitcoinModal" 
         x-transition:enter="transition ease-out duration-300"
         x-transition:enter-start="opacity-0"
         x-transition:enter-end="opacity-100"
         class="fixed inset-0 z-50 flex items-center justify-center p-4 bg-black/80"
         @click.self="showBitcoinModal = false">
        <div class="glass-effect rounded-2xl p-6 max-w-md w-full text-center">
            <div class="w-20 h-20 mx-auto mb-4 rounded-full bg-gradient-to-br from-orange-400 to-orange-600 flex items-center justify-center">
                <span class="text-white font-black text-3xl">₿</span>
            </div>
            <h3 class="text-xl font-bold text-white mb-2">Pago con Bitcoin</h3>
            <p class="text-gray-400 mb-4">Envía <span class="text-orange-400 font-bold" x-text="getBtcAmount()"></span> BTC a:</p>
            
            <div class="p-4 rounded-lg bg-gray-800 border border-orange-500/30 mb-4">
                <div class="w-32 h-32 mx-auto mb-3 bg-white rounded-lg flex items-center justify-center">
                    <span class="text-4xl">📱</span>
                </div>
                <p class="text-xs font-mono text-orange-400 break-all">bc1qxy2kgdygjrsqtzq2n0yrf2493p83kkfjhx0wlh</p>
            </div>
            
            <p class="text-sm text-gray-400 mb-4">
                ⏱️ Esta dirección expira en <span class="text-orange-400 font-bold">15:00</span> minutos
            </p>
            
            <button @click="confirmarPagoBitcoin()" :disabled="procesandoPago"
                    class="w-full py-3 rounded-xl font-bold bg-gradient-to-r from-orange-500 to-orange-600 text-white hover:from-orange-400 hover:to-orange-500 disabled:opacity-50">
                <span x-show="!procesandoPago">Ya realicé el pago</span>
                <span x-show="procesandoPago">Verificando...</span>
            </button>
            
            <button @click="showBitcoinModal = false" class="w-full mt-2 py-2 text-gray-400 hover:text-white">
                Cancelar
            </button>
        </div>
    </div>

    <!-- Modal de éxito -->
    <div x-cloak x-show="showSuccessModal" 
         x-transition:enter="transition ease-out duration-300"
         x-transition:enter-start="opacity-0"
         x-transition:enter-end="opacity-100"
         class="fixed inset-0 z-50 flex items-center justify-center p-4 bg-black/80">
        <div class="glass-effect rounded-2xl p-8 max-w-md w-full text-center animate-bounce-ball">
            <div class="w-24 h-24 mx-auto mb-4 rounded-full bg-gradient-to-br from-green-400 to-emerald-500 flex items-center justify-center shadow-lg shadow-green-500/50">
                <svg class="w-12 h-12 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="3" d="M5 13l4 4L19 7"/>
                </svg>
            </div>
            <h3 class="text-2xl font-black text-green-400 mb-2">¡Recarga Exitosa!</h3>
            <p class="text-gray-400 mb-4">Tu saldo ha sido actualizado</p>
            <p class="text-4xl font-black text-white mb-6">+$<span x-text="montoRecargado.toLocaleString()"></span></p>
            
            <a href="{{ route('bingo.index') }}" 
               class="inline-block w-full py-3 rounded-xl font-bold bg-gradient-to-r from-green-500 to-emerald-600 text-white hover:from-green-400 hover:to-emerald-500">
                ¡A Jugar! 🎰
            </a>
        </div>
    </div>
</div>

@push('scripts')
<script>
function recargaPage() {
    return {
        montosPredefinidos: [10000, 25000, 50000, 100000],
        montoSeleccionado: null,
        montoPersonalizado: '',
        metodoPago: null,
        procesando: false,
        procesandoPago: false,
        showStripeModal: false,
        showPaypalModal: false,
        showBitcoinModal: false,
        showSuccessModal: false,
        montoRecargado: 0,
        tarjeta: {
            numero: '',
            vencimiento: '',
            cvv: '',
            nombre: ''
        },

        getMontoFinal() {
            if (this.montoSeleccionado) return this.montoSeleccionado;
            if (this.montoPersonalizado) return parseInt(this.montoPersonalizado) || 0;
            return 0;
        },

        getBonus() {
            const monto = this.getMontoFinal();
            if (monto >= 100000) return 10;
            if (monto >= 50000) return 5;
            if (monto >= 25000) return 2;
            return 0;
        },

        getMetodoNombre() {
            return {
                'stripe': 'Tarjeta',
                'paypal': 'PayPal',
                'bitcoin': 'Bitcoin'
            }[this.metodoPago] || '';
        },

        getBtcAmount() {
            // Simulación: 1 BTC = $60,000 USD aprox
            const btcPrice = 60000000; // en pesos
            return (this.getMontoFinal() / btcPrice).toFixed(6);
        },

        canProceed() {
            return this.getMontoFinal() >= 1000 && this.metodoPago;
        },

        formatearTarjeta() {
            let value = this.tarjeta.numero.replace(/\s/g, '').replace(/\D/g, '');
            let formatted = value.match(/.{1,4}/g)?.join(' ') || value;
            this.tarjeta.numero = formatted.substring(0, 19);
        },

        procesarPago() {
            if (!this.canProceed()) return;
            
            this.procesando = true;
            
            setTimeout(() => {
                this.procesando = false;
                
                if (this.metodoPago === 'stripe') {
                    this.showStripeModal = true;
                } else if (this.metodoPago === 'paypal') {
                    this.showPaypalModal = true;
                } else if (this.metodoPago === 'bitcoin') {
                    this.showBitcoinModal = true;
                }
            }, 500);
        },

        async confirmarPagoStripe() {
            this.procesandoPago = true;
            
            // Simular procesamiento
            await this.realizarRecarga('Tarjeta (Stripe)');
        },

        async confirmarPagoPaypal() {
            this.procesandoPago = true;
            
            // Simular procesamiento
            await this.realizarRecarga('PayPal');
        },

        async confirmarPagoBitcoin() {
            this.procesandoPago = true;
            
            // Simular procesamiento
            await this.realizarRecarga('Bitcoin');
        },

        async realizarRecarga(metodo) {
            try {
                const response = await fetch('{{ route("bingo.procesarRecarga") }}', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content,
                        'Accept': 'application/json'
                    },
                    body: JSON.stringify({
                        monto: this.getMontoFinal(),
                        metodo: metodo
                    })
                });
                
                const data = await response.json();
                
                if (data.success) {
                    this.montoRecargado = data.monto;
                    this.showStripeModal = false;
                    this.showPaypalModal = false;
                    this.showBitcoinModal = false;
                    this.showSuccessModal = true;
                } else {
                    alert(data.error || 'Error al procesar el pago');
                }
            } catch (error) {
                console.error('Error:', error);
                alert('Error de conexión');
            } finally {
                this.procesandoPago = false;
            }
        }
    }
}
</script>
@endpush
@endsection
