<?php

namespace App\Http\Controllers;

use App\Models\Juego;
use App\Models\Carton;
use App\Models\User;
use App\Events\BolaSorteada;
use App\Events\JuegoActualizado;
use App\Events\SorteoIniciado;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Session;

class BingoController extends Controller
{
    const BALANCE_INICIAL = 100000;

    private function getBalance(): float
    {
        $this->procesarReembolsoPendiente();
        
        if (Auth::check()) {
            return (float) Auth::user()->balance;
        }
        if (!Session::has('balance')) {
            Session::put('balance', self::BALANCE_INICIAL);
        }
        return (float) Session::get('balance');
    }

    private function procesarReembolsoPendiente(): void
    {
        $sessionId = Session::getId();
        $reembolso = cache()->get('reembolso_pendiente_' . $sessionId, 0);
        
        if ($reembolso > 0) {
            if (Auth::check()) {
                Auth::user()->increment('balance', $reembolso);
            } else {
                $balanceActual = Session::get('balance', self::BALANCE_INICIAL);
                Session::put('balance', $balanceActual + $reembolso);
            }
            
            cache()->forget('reembolso_pendiente_' . $sessionId);
            Session::flash('reembolso', $reembolso);
        }
    }

    private function updateBalance(float $cantidad): float
    {
        if (Auth::check()) {
            $user = Auth::user();
            $user->balance = $user->balance + $cantidad;
            $user->save();
            return (float) $user->balance;
        }
        $balanceActual = $this->getBalance();
        $nuevoBalance = $balanceActual + $cantidad;
        Session::put('balance', $nuevoBalance);
        return $nuevoBalance;
    }

    private function getGanancias(): array
    {
        if (!Session::has('ganancias')) {
            Session::put('ganancias', []);
        }
        return Session::get('ganancias');
    }

    private function addGanancia(string $tipo, float $monto): void
    {
        $ganancias = $this->getGanancias();
        $ganancias[] = [
            'tipo' => $tipo,
            'monto' => $monto,
            'fecha' => now()->format('H:i:s'),
        ];
        Session::put('ganancias', $ganancias);
    }

    public function index()
    {
        $juegoActivo = Juego::whereIn('estado', ['esperando', 'en_curso'])->first();
        
        if (!$juegoActivo) {
            $juegoActivo = Juego::create([
                'precio_carton' => 1000,
                'acumulado' => 50000,
            ]);
        }

        $user = Auth::user();
        $misCartones = Carton::where('juego_id', $juegoActivo->id)
            ->where('user_id', $user->id)
            ->count();

        $balance = $this->getBalance();
        $ganancias = $this->getGanancias();

        return view('bingo.index', compact('juegoActivo', 'misCartones', 'balance', 'ganancias', 'user'));
    }

    public function jugar(Juego $juego)
    {
        if ($juego->estado === 'finalizado') {
            return redirect()->route('bingo.index')
                ->with('error', 'Este juego ya finalizó. ¡Comienza uno nuevo!');
        }

        $user = Auth::user();
        
        $cartones = Carton::where('juego_id', $juego->id)
            ->where('user_id', $user->id)
            ->get();

        $bolasSorteadas = $juego->bolasSorteadas()->get();
        $ultimaBola = $juego->getUltimaBola();

        $cartonesJson = $cartones->map(function ($c) {
            return [
                'id' => $c->id,
                'numeros' => $c->numeros,
                'marcados' => $c->marcados ?? [],
                'tiene_linea' => $c->tiene_linea,
                'tiene_bingo' => $c->tiene_bingo,
                'posicion' => $c->posicion,
                'apuesta' => (int) $c->apuesta,
            ];
        });

        $historialJson = $bolasSorteadas->map(function ($b) {
            return [
                'numero' => $b->numero,
                'letra' => $b->letra,
                'orden' => $b->orden,
                'color' => $b->getColorClass(),
            ];
        });

        $ultimaBolaJson = $ultimaBola ? [
            'numero' => $ultimaBola->numero,
            'letra' => $ultimaBola->letra,
            'color' => $ultimaBola->getColorClass(),
            'display' => $ultimaBola->getDisplay(),
        ] : null;

        $numerosSorteados = $bolasSorteadas->pluck('numero')->toArray();

        $balance = $this->getBalance();
        $ganancias = $this->getGanancias();
        
        $nombreJugador = $user->name;
        $jugadoresActivos = $this->getJugadoresActivos($juego, $user->id);

        return view('bingo.jugar', compact(
            'juego', 
            'cartones', 
            'bolasSorteadas', 
            'ultimaBola',
            'cartonesJson',
            'historialJson',
            'ultimaBolaJson',
            'numerosSorteados',
            'balance',
            'ganancias',
            'nombreJugador',
            'jugadoresActivos',
            'user'
        ));
    }

    private function getJugadoresActivos(Juego $juego, ?int $excluirUserId = null): array
    {
        $query = $juego->cartones()
            ->select('user_id')
            ->selectRaw('COUNT(*) as total_cartones')
            ->whereNotNull('user_id')
            ->groupBy('user_id');

        if ($excluirUserId) {
            $query->where('user_id', '!=', $excluirUserId);
        }

        $cartonesPorUsuario = $query->get();

        $jugadores = [];
        foreach ($cartonesPorUsuario as $item) {
            $usuario = User::find($item->user_id);
            $jugadores[] = [
                'user_id' => $item->user_id,
                'nombre' => $usuario ? $usuario->name : 'Jugador',
                'cartones' => $item->total_cartones,
            ];
        }

        return $jugadores;
    }

    public function comprarCartones(Request $request, Juego $juego)
    {
        if ($juego->estado === 'finalizado') {
            return redirect()->route('bingo.index')
                ->with('error', 'Este juego ya finalizó. ¡Comienza uno nuevo!');
        }

        $request->validate([
            'cantidad' => 'required|integer|min:1|max:4',
            'apuesta' => 'required|numeric|min:1',
        ]);

        $user = Auth::user();
        $cartones = [];

        $cartonesExistentes = Carton::where('juego_id', $juego->id)
            ->where('user_id', $user->id)
            ->count();

        $disponibles = 4 - $cartonesExistentes;
        
        if ($disponibles <= 0) {
            return back()->with('error', 'Ya tienes el máximo de 4 cartones');
        }
        
        $cantidadComprar = min($request->cantidad, $disponibles);

        $costoTotal = $cantidadComprar * $request->apuesta * $juego->precio_carton;
        $balance = $this->getBalance();

        if ($balance < $costoTotal) {
            return back()->with('error', "Saldo insuficiente. Necesitas $" . number_format($costoTotal, 0) . " pero tienes $" . number_format($balance, 0));
        }

        $nuevoBalance = $this->updateBalance(-$costoTotal);

        for ($i = 0; $i < $cantidadComprar; $i++) {
            $cartones[] = Carton::create([
                'juego_id' => $juego->id,
                'user_id' => $user->id,
                'session_id' => Session::getId(),
                'numeros' => Carton::generarNumeros(),
                'marcados' => [],
                'apuesta' => $request->apuesta,
                'posicion' => $cartonesExistentes + $i + 1,
            ]);
        }

        $juego->increment('acumulado', $costoTotal * 0.7);
        $juego->increment('premio_linea', $costoTotal * 0.1);
        $juego->increment('premio_bingo', $costoTotal * 0.15);

        $juego->refresh();
        
        $totalCartones = $juego->cartones()->count();
        $totalJugadores = $juego->cartones()->distinct('user_id')->count('user_id');
        $jugadoresActivos = $this->getJugadoresActivos($juego);

        broadcast(new JuegoActualizado(
            $juego->id,
            'carton_comprado',
            (float) $juego->acumulado,
            (float) $juego->premio_linea,
            (float) $juego->premio_bingo,
            $totalCartones,
            $totalJugadores,
            $juego->estado,
            $jugadoresActivos,
            $user->name
        ))->toOthers();

        return redirect()->route('bingo.jugar', $juego)
            ->with('success', "¡{$cantidadComprar} cartón(es) comprado(s)! Gastaste $" . number_format($costoTotal, 0));
    }

    public function sortearBola(Request $request, Juego $juego)
    {
        $esPrimeraBola = $juego->estado === 'esperando';
        
        if ($esPrimeraBola) {
            $juego->iniciar();
            $nombreJugador = Auth::user()->name;
            broadcast(new SorteoIniciado($juego->id, 'en_curso', $nombreJugador));
        }

        if ($juego->estado === 'finalizado') {
            return response()->json([
                'error' => 'El juego ha finalizado',
                'finalizado' => true,
                'limite_alcanzado' => $juego->haAlcanzadoLimite(),
                'redirect' => route('bingo.index'),
            ], 400);
        }

        $bola = $juego->sortearBola();

        if (!$bola) {
            $juego->finalizar();
            $mensaje = $juego->haAlcanzadoLimite() 
                ? 'Se alcanzó el límite de ' . $juego->getMaxBolas() . ' bolas. Juego finalizado.' 
                : 'No hay más bolas';
            return response()->json(['error' => $mensaje, 'finalizado' => true, 'limite_alcanzado' => true], 400);
        }

        $this->actualizarCartones($juego, $bola->numero);

        $ganadorLinea = $this->verificarGanadorLinea($juego);
        $ganadorBingo = $this->verificarGanadorBingo($juego);

        $juego->refresh();
        $juegoFinalizado = $ganadorBingo !== null || $juego->haAlcanzadoLimite();
        
        if ($juego->haAlcanzadoLimite() && !$ganadorBingo) {
            $juego->finalizar();
            $juego->refresh();
        }

        $bolaData = [
            'numero' => $bola->numero,
            'letra' => $bola->letra,
            'orden' => $bola->orden,
            'color' => $bola->getColorClass(),
            'display' => $bola->getDisplay(),
        ];

        broadcast(new BolaSorteada(
            $juego->id,
            $bolaData,
            $juego->bolasSorteadas()->count(),
            $juego->getMaxBolas(),
            $ganadorLinea,
            $ganadorBingo,
            $juegoFinalizado,
            $juego->haAlcanzadoLimite(),
            (float) $juego->acumulado,
            (float) $juego->premio_linea,
            (float) $juego->premio_bingo,
            $juego->estado
        ))->toOthers();

        return response()->json([
            'bola' => $bolaData,
            'bolas_sorteadas' => $juego->bolasSorteadas()->count(),
            'max_bolas' => $juego->getMaxBolas(),
            'ganador_linea' => $ganadorLinea,
            'ganador_bingo' => $ganadorBingo,
            'finalizado' => $juegoFinalizado,
            'limite_alcanzado' => $juego->haAlcanzadoLimite(),
            'balance' => $this->getBalance(),
        ]);
    }

    private function actualizarCartones(Juego $juego, int $numero): void
    {
        $cartones = $juego->cartones;
        
        foreach ($cartones as $carton) {
            $carton->marcarNumero($numero);
        }
    }

    private function verificarGanadorLinea(Juego $juego)
    {
        if ($juego->ganador_linea_id) {
            return null;
        }

        $user = Auth::user();

        foreach ($juego->cartones as $carton) {
            if (!$carton->tiene_linea && $carton->verificarLinea()) {
                $carton->update(['tiene_linea' => true]);
                $juego->update(['ganador_linea_id' => $carton->user_id]);
                
                $premioBase = $juego->premio_linea;
                $premioTotal = $premioBase * $carton->apuesta;
                
                $esDelJugador = $carton->user_id === $user->id;
                if ($esDelJugador) {
                    $this->updateBalance($premioTotal);
                    $this->addGanancia('Línea', $premioTotal);
                }
                
                $ganadorUser = User::find($carton->user_id);
                $nombreGanador = $ganadorUser ? $ganadorUser->name : 'Jugador';
                
                return [
                    'carton_id' => $carton->id,
                    'premio' => $premioTotal,
                    'es_mio' => $esDelJugador,
                    'nombre' => $nombreGanador,
                ];
            }
        }

        return null;
    }

    private function verificarGanadorBingo(Juego $juego)
    {
        $user = Auth::user();

        foreach ($juego->cartones as $carton) {
            if (!$carton->tiene_bingo && $carton->verificarBingo()) {
                $carton->update(['tiene_bingo' => true]);
                $juego->update(['ganador_bingo_id' => $carton->user_id]);
                $juego->finalizar();
                
                $premioBase = $juego->premio_bingo + $juego->acumulado;
                $premioTotal = $premioBase * $carton->apuesta;
                
                $esDelJugador = $carton->user_id === $user->id;
                if ($esDelJugador) {
                    $this->updateBalance($premioTotal);
                    $this->addGanancia('BINGO', $premioTotal);
                }
                
                $ganadorUser = User::find($carton->user_id);
                $nombreGanador = $ganadorUser ? $ganadorUser->name : 'Jugador';
                
                return [
                    'carton_id' => $carton->id,
                    'premio' => $premioTotal,
                    'es_mio' => $esDelJugador,
                    'nombre' => $nombreGanador,
                ];
            }
        }

        return null;
    }

    public function obtenerEstado(Juego $juego)
    {
        $user = Auth::user();
        
        $cartones = Carton::where('juego_id', $juego->id)
            ->where('user_id', $user->id)
            ->get();

        return response()->json([
            'juego' => [
                'estado' => $juego->estado,
                'acumulado' => $juego->acumulado,
                'premio_linea' => $juego->premio_linea,
                'premio_bingo' => $juego->premio_bingo,
                'bolas_sorteadas' => $juego->bolasSorteadas()->count(),
            ],
            'cartones' => $cartones->map(function ($carton) {
                return [
                    'id' => $carton->id,
                    'numeros' => $carton->numeros,
                    'marcados' => $carton->marcados ?? [],
                    'tiene_linea' => $carton->tiene_linea,
                    'tiene_bingo' => $carton->tiene_bingo,
                    'posicion' => $carton->posicion,
                ];
            }),
            'bolas' => $juego->bolasSorteadas->map(function ($bola) {
                return [
                    'numero' => $bola->numero,
                    'letra' => $bola->letra,
                    'color' => $bola->getColorClass(),
                ];
            }),
        ]);
    }

    public function reiniciar()
    {
        Juego::whereIn('estado', ['esperando', 'en_curso'])->update(['estado' => 'finalizado', 'fin_at' => now()]);
        
        $juego = Juego::create([
            'precio_carton' => 1000,
            'acumulado' => 50000,
        ]);

        return redirect()->route('bingo.index')->with('success', 'Nuevo juego iniciado');
    }

    public function abandonarJuego(Juego $juego)
    {
        $user = Auth::user();
        $nombreJugador = $user->name;
        
        $cartonesEliminados = Carton::where('juego_id', $juego->id)
            ->where('user_id', $user->id)
            ->delete();
        
        $totalJugadores = $juego->cartones()->distinct('user_id')->count('user_id');
        $totalCartones = $juego->cartones()->count();
        
        broadcast(new \App\Events\JugadorSalio(
            $juego->id,
            $nombreJugador,
            $totalJugadores,
            $totalCartones
        ));

        return redirect()->route('bingo.index')->with('success', 'Has abandonado el juego. ¡Vuelve pronto!');
    }

    public function iniciarSorteoAuto(Juego $juego)
    {
        if ($juego->estado === 'finalizado') {
            return response()->json(['error' => 'Juego finalizado'], 400);
        }

        $sorteoActivo = cache()->get('sorteo_auto_' . $juego->id);
        
        if ($sorteoActivo && isset($sorteoActivo['es_admin']) && $sorteoActivo['es_admin']) {
            if (!Session::get('is_admin')) {
                return response()->json([
                    'error' => 'El moderador controla el sorteo',
                    'controlador' => $sorteoActivo['nombre'],
                ], 403);
            }
        }

        $user = Auth::user();
        $esAdmin = Session::get('is_admin', false);
        
        cache()->put('sorteo_auto_' . $juego->id, [
            'user_id' => $user->id,
            'nombre' => $esAdmin ? '🎰 MODERADOR' : $user->name,
            'iniciado_at' => now(),
            'es_admin' => $esAdmin,
        ], now()->addHours(1));

        if ($juego->estado === 'esperando') {
            $juego->iniciar();
        }

        broadcast(new \App\Events\SorteoAutoIniciado(
            $juego->id,
            $user->id,
            $esAdmin ? '🎰 MODERADOR' : $user->name,
            $juego->estado
        ));

        return response()->json([
            'success' => true,
            'mensaje' => 'Sorteo automático iniciado',
            'controlador' => $esAdmin ? '🎰 MODERADOR' : $user->name,
        ]);
    }

    public function detenerSorteoAuto(Juego $juego)
    {
        $user = Auth::user();
        $sorteoActivo = cache()->get('sorteo_auto_' . $juego->id);
        $esAdmin = Session::get('is_admin', false);
        
        if ($sorteoActivo && !$esAdmin && $sorteoActivo['user_id'] !== $user->id) {
            return response()->json(['error' => 'Solo el controlador puede detener el sorteo'], 403);
        }

        cache()->forget('sorteo_auto_' . $juego->id);

        broadcast(new \App\Events\SorteoAutoDetenido($juego->id, $user->id));

        return response()->json([
            'success' => true,
            'mensaje' => 'Sorteo automático detenido',
        ]);
    }

    public function estadoSorteoAuto(Juego $juego)
    {
        $sorteoActivo = cache()->get('sorteo_auto_' . $juego->id);
        $user = Auth::user();
        $esAdmin = Session::get('is_admin', false);
        
        return response()->json([
            'activo' => $sorteoActivo !== null,
            'es_controlador' => $sorteoActivo && ($sorteoActivo['user_id'] === $user->id || $esAdmin),
            'controlador' => $sorteoActivo ? $sorteoActivo['nombre'] : null,
            'es_admin' => $sorteoActivo && isset($sorteoActivo['es_admin']) && $sorteoActivo['es_admin'],
        ]);
    }

    public function multiplicarApuesta(Request $request, Juego $juego)
    {
        if ($juego->estado !== 'esperando') {
            return response()->json(['error' => 'El sorteo ya comenzó', 'apuestas_cerradas' => true], 400);
        }

        $request->validate([
            'multiplicador' => 'required|integer|min:1|max:50',
            'carton_id' => 'required|integer|exists:cartones,id',
        ]);

        $user = Auth::user();
        $nuevaApuesta = $request->multiplicador;

        $carton = Carton::where('id', $request->carton_id)
            ->where('juego_id', $juego->id)
            ->where('user_id', $user->id)
            ->first();

        if (!$carton) {
            return response()->json(['error' => 'Cartón no encontrado'], 404);
        }

        $apuestaActual = $carton->apuesta;
        
        if ($nuevaApuesta == $apuestaActual) {
            return response()->json([
                'success' => true,
                'carton_id' => $carton->id,
                'nueva_apuesta' => $apuestaActual,
                'acumulado' => $juego->acumulado,
                'premio_linea' => $juego->premio_linea,
                'premio_bingo' => $juego->premio_bingo,
            ]);
        }

        $diferencia = $nuevaApuesta - $apuestaActual;
        $totalAdicional = $diferencia * $juego->precio_carton;
        
        if ($totalAdicional > 0) {
            $balance = $this->getBalance();
            if ($balance < $totalAdicional) {
                return response()->json([
                    'error' => 'Saldo insuficiente para aumentar la apuesta',
                    'balance' => $balance,
                ], 400);
            }
            $this->updateBalance(-$totalAdicional);
        } else {
            $this->updateBalance(abs($totalAdicional));
        }
        
        $carton->update(['apuesta' => $nuevaApuesta]);

        if ($totalAdicional > 0) {
            $juego->increment('acumulado', $totalAdicional * 0.7);
            $juego->increment('premio_linea', $totalAdicional * 0.1);
            $juego->increment('premio_bingo', $totalAdicional * 0.15);
        } else {
            $juego->decrement('acumulado', abs($totalAdicional) * 0.7);
            $juego->decrement('premio_linea', abs($totalAdicional) * 0.1);
            $juego->decrement('premio_bingo', abs($totalAdicional) * 0.15);
        }

        $juego->refresh();

        $totalCartones = $juego->cartones()->count();
        $totalJugadores = $juego->cartones()->distinct('user_id')->count('user_id');

        broadcast(new JuegoActualizado(
            $juego->id,
            'apuesta_modificada',
            (float) $juego->acumulado,
            (float) $juego->premio_linea,
            (float) $juego->premio_bingo,
            $totalCartones,
            $totalJugadores,
            $juego->estado,
            $this->getJugadoresActivos($juego)
        ))->toOthers();

        return response()->json([
            'success' => true,
            'carton_id' => $carton->id,
            'nueva_apuesta' => $nuevaApuesta,
            'acumulado' => $juego->acumulado,
            'premio_linea' => $juego->premio_linea,
            'premio_bingo' => $juego->premio_bingo,
            'balance' => $this->getBalance(),
        ]);
    }

    public function ajustarApuesta(Request $request, Juego $juego)
    {
        if ($juego->estado !== 'esperando') {
            return response()->json(['error' => 'El sorteo ya comenzó', 'apuestas_cerradas' => true], 400);
        }

        $request->validate([
            'ajuste' => 'required|integer|min:-10|max:10',
            'carton_id' => 'required|integer|exists:cartones,id',
        ]);

        $user = Auth::user();
        $ajuste = $request->ajuste;

        $carton = Carton::where('id', $request->carton_id)
            ->where('juego_id', $juego->id)
            ->where('user_id', $user->id)
            ->first();

        if (!$carton) {
            return response()->json(['error' => 'Cartón no encontrado'], 404);
        }

        $apuestaActual = $carton->apuesta;
        $nuevaApuesta = max(1, $apuestaActual + $ajuste);
        
        if ($nuevaApuesta === $apuestaActual) {
            return response()->json([
                'success' => true,
                'carton_id' => $carton->id,
                'nueva_apuesta' => $apuestaActual,
                'acumulado' => $juego->acumulado,
                'premio_linea' => $juego->premio_linea,
                'premio_bingo' => $juego->premio_bingo,
            ]);
        }

        $diferencia = $nuevaApuesta - $apuestaActual;
        $totalAdicional = $diferencia * $juego->precio_carton;
        
        if ($totalAdicional > 0) {
            $balance = $this->getBalance();
            if ($balance < $totalAdicional) {
                return response()->json([
                    'error' => 'Saldo insuficiente',
                    'balance' => $balance,
                ], 400);
            }
            $this->updateBalance(-$totalAdicional);
        } else {
            $this->updateBalance(abs($totalAdicional));
        }
        
        $carton->update(['apuesta' => $nuevaApuesta]);

        if ($totalAdicional > 0) {
            $juego->increment('acumulado', $totalAdicional * 0.7);
            $juego->increment('premio_linea', $totalAdicional * 0.1);
            $juego->increment('premio_bingo', $totalAdicional * 0.15);
        } else {
            $juego->decrement('acumulado', abs($totalAdicional) * 0.7);
            $juego->decrement('premio_linea', abs($totalAdicional) * 0.1);
            $juego->decrement('premio_bingo', abs($totalAdicional) * 0.15);
        }

        $juego->refresh();

        $totalCartones = $juego->cartones()->count();
        $totalJugadores = $juego->cartones()->distinct('user_id')->count('user_id');

        broadcast(new JuegoActualizado(
            $juego->id,
            'apuesta_ajustada',
            (float) $juego->acumulado,
            (float) $juego->premio_linea,
            (float) $juego->premio_bingo,
            $totalCartones,
            $totalJugadores,
            $juego->estado,
            $this->getJugadoresActivos($juego)
        ))->toOthers();

        return response()->json([
            'success' => true,
            'carton_id' => $carton->id,
            'nueva_apuesta' => $nuevaApuesta,
            'acumulado' => $juego->acumulado,
            'premio_linea' => $juego->premio_linea,
            'premio_bingo' => $juego->premio_bingo,
            'balance' => $this->getBalance(),
        ]);
    }

    public function reiniciarBalance()
    {
        if (Auth::check()) {
            $user = Auth::user();
            $user->balance = self::BALANCE_INICIAL;
            $user->save();
        } else {
            Session::put('balance', self::BALANCE_INICIAL);
        }
        Session::put('ganancias', []);
        Session::put('transacciones', []);
        return redirect()->route('bingo.index')->with('success', '¡Balance reiniciado a $' . number_format(self::BALANCE_INICIAL, 0) . '!');
    }

    private function getTransacciones(): array
    {
        if (!Session::has('transacciones')) {
            Session::put('transacciones', []);
        }
        return Session::get('transacciones');
    }

    private function addTransaccion(string $tipo, float $monto, ?string $metodo = null): void
    {
        $transacciones = $this->getTransacciones();
        $transacciones[] = [
            'tipo' => $tipo,
            'monto' => $monto,
            'metodo' => $metodo,
            'fecha' => now()->format('d/m H:i'),
        ];
        Session::put('transacciones', $transacciones);
    }

    public function recargar()
    {
        $balance = $this->getBalance();
        $transacciones = $this->getTransacciones();
        $user = Auth::user();
        
        return view('bingo.recargar', compact('balance', 'transacciones', 'user'));
    }

    public function procesarRecarga(Request $request)
    {
        $request->validate([
            'monto' => 'required|numeric|min:1000',
            'metodo' => 'required|string',
        ]);

        $monto = (float) $request->monto;
        $metodo = $request->metodo;

        $bonus = 0;
        if ($monto >= 100000) {
            $bonus = $monto * 0.10;
        } elseif ($monto >= 50000) {
            $bonus = $monto * 0.05;
        } elseif ($monto >= 25000) {
            $bonus = $monto * 0.02;
        }

        $montoTotal = $monto + $bonus;

        $nuevoBalance = $this->updateBalance($montoTotal);

        $this->addTransaccion('recarga', $montoTotal, $metodo);

        return response()->json([
            'success' => true,
            'monto' => $montoTotal,
            'bonus' => $bonus,
            'balance' => $nuevoBalance,
            'mensaje' => '¡Recarga exitosa!',
        ]);
    }
}
