<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Juego;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Laravel\Socialite\Facades\Socialite;

class AuthController extends Controller
{
    public function landing()
    {
        if (Auth::check()) {
            return redirect()->route('bingo.index');
        }

        $juegoActivo = Juego::whereIn('estado', ['esperando', 'en_curso'])->first();
        $acumulado = $juegoActivo ? $juegoActivo->acumulado : 50000;
        $totalJugadores = $juegoActivo ? $juegoActivo->cartones()->distinct('session_id')->count('session_id') : 0;

        return view('auth.landing', compact('acumulado', 'totalJugadores'));
    }

    public function showLogin()
    {
        if (Auth::check()) {
            return redirect()->route('bingo.index');
        }
        return view('auth.login');
    }

    public function showRegister()
    {
        if (Auth::check()) {
            return redirect()->route('bingo.index');
        }
        return view('auth.register');
    }

    public function register(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email',
            'password' => 'required|min:6|confirmed',
        ]);

        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
            'balance' => 100000,
        ]);

        Auth::login($user);

        return redirect()->route('bingo.index')->with('success', '¡Bienvenido! Tu cuenta ha sido creada con $100,000 de saldo inicial.');
    }

    public function login(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
            'password' => 'required',
        ]);

        if (Auth::attempt($request->only('email', 'password'), $request->remember)) {
            $request->session()->regenerate();
            return redirect()->intended(route('bingo.index'))->with('success', '¡Bienvenido de nuevo!');
        }

        return back()->withErrors([
            'email' => 'Las credenciales no coinciden con nuestros registros.',
        ])->withInput($request->only('email'));
    }

    public function logout(Request $request)
    {
        Auth::logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect()->route('landing')->with('success', '¡Hasta pronto!');
    }

    public function redirectToGoogle()
    {
        return Socialite::driver('google')->redirect();
    }

    public function handleGoogleCallback()
    {
        try {
            $googleUser = Socialite::driver('google')->user();

            $user = User::where('google_id', $googleUser->id)
                ->orWhere('email', $googleUser->email)
                ->first();

            if ($user) {
                if (!$user->google_id) {
                    $user->update([
                        'google_id' => $googleUser->id,
                        'avatar' => $googleUser->avatar,
                    ]);
                }
            } else {
                $user = User::create([
                    'name' => $googleUser->name,
                    'email' => $googleUser->email,
                    'google_id' => $googleUser->id,
                    'avatar' => $googleUser->avatar,
                    'balance' => 100000,
                ]);
            }

            Auth::login($user);

            return redirect()->route('bingo.index')->with('success', '¡Bienvenido ' . $user->name . '!');

        } catch (\Exception $e) {
            return redirect()->route('login')->with('error', 'Error al iniciar sesión con Google. Intenta de nuevo.');
        }
    }
}
